from art_export import ArtExporter
import os

# Exporter for Amstrad CPC line
# Supports standard and full charset (including control codes characters)
# Supports all 3 modes depending on character width (0: 20, 1: 40, 2: 80)

# put that file in the "formats" folder inside the Playscii directory


# Credits
# Main code: Logiker
# Initial code exporter: emptyfortress
# Initial code cpc viewer: DeadSystems


#returns AMSDOS header (array with 128 bytes)
#see also: https://www.cpcwiki.eu/index.php/AMSDOS_Header
#Author: Logiker
def get_AMSDOS_header(file_name, file_type, file_len, load_addr, start_addr):
    file_name = file_name[:7]
    #file_name too long?
    if len(file_name)>8:
        return 1
    #file_type supported?
    if (file_type not in ('bas', 'bin')):
        return 2
    ret = bytearray()
    ret.append(0x00)
    ret += bytearray(file_name.encode())
    #missing spaces
    ret += bytearray([0x20] * (8-len(file_name)*1))
    ret += bytearray(file_type.encode())
    ret += bytearray([0x00] * 4)
    ret.append(0x00)    #tape only
    ret.append(0x00)    #tape only
    if file_type=='bas':
        ret.append(0x00)
    else:   #bin
        ret.append(0x02)
    ret += bytearray(b'\x00\x00')    #addr of 2k buffer?
    ret += load_addr.to_bytes(2, 'little')
    ret.append(0x00)
    ret += file_len.to_bytes(2, 'little')
    ret += start_addr.to_bytes(2, 'little')
    ret += bytearray([0x00] * 36)   #unused
    ret += file_len.to_bytes(3, 'little')   #Real length,	24-bit number. Just a copy, not used!
    check_sum = sum(ret)
    ret += check_sum.to_bytes(2, 'little')  #Checksum,	    Unsigned sum of all bytes until this field
    ret += bytearray([0x00] * 59)   #unused
    
    return ret

    
class CPCBinExporter(ArtExporter):
    format_name = 'CPC'
    format_description = """
Creates an executable that runs on Amstrad CPC computers.
Assumes single frame, single layer document.
CPC charset and palette should be used.
    """
    file_extension = 'bin'
    inks = []
    ink_count = 0
    max_col = 4
    cpc_mode = 1

    def set_cpc_mode(self):
        if self.art.width==20:
            self.cpc_mode = 0
            self.max_col = 16
        if self.art.width==40:
            self.cpc_mode = 1
            self.max_col = 4
        if self.art.width==80:
            self.cpc_mode = 2
            self.max_col = 2

    def get_ink_for_col(self, col, out_data, x, y):
        if col==0:  #transparent color will be seen as white
            print("Attention: Transparent color will be seen as white!")
            col=27
        if col in self.inks:
            return self.inks.index(col)
        if len(self.inks)>=self.max_col:        #4 inks for mode 1, addapt for other modes
            return self.max_col-1
        #add ink
        self.inks.append(col)
        ink_cmd = 28
        out_data += (ink_cmd).to_bytes(1, 'big')
        out_data += (len(self.inks)-1).to_bytes(1, 'big')
        out_data += (col-1).to_bytes(1, 'big')
        out_data += (col-1).to_bytes(1, 'big')
        return len(self.inks)-1

    def run_export(self, out_filename, options):
        out_data = bytearray()
        
        if self.art.charset.name=='cpc':    #only the standard charset is not 'full'
            full_charset = False        
        else:                               #other charsets are full (also valid for mode 0 or mode 2)
            full_charset = True
        
        set_screen_mode, pen, paper, ink, end_byte = 4, 15, 14, 28, 31 # ascii codes controle to pen and paper change, see also: https://handwiki.org/wiki/Amstrad_CPC_character_set
        oldbg, oldfg, ink_bg, ink_fg = 100, 100, 100, 100

        self.set_cpc_mode()
        out_data += (set_screen_mode).to_bytes(1, 'big')
        out_data += (self.cpc_mode).to_bytes(1, 'big')

        for y in range(self.art.height):
            for x in range(self.art.width):
                # fg: foreground colour, bg: background colour
                char, fg, bg, _ = self.art.get_tile_at(0, 0, x, y)
                if char != 0:
                    ink_fg = self.get_ink_for_col(fg, out_data,x,y)
                ink_bg = self.get_ink_for_col(bg, out_data,x,y)
                
                if oldbg != ink_bg : # background changed, write new if true
                    out_data += paper.to_bytes(1, 'big')
                    out_data += ink_bg.to_bytes(1, 'big')
                    oldbg = ink_bg
                if oldfg != ink_fg : # foreground changed, write new if true
                    out_data += pen.to_bytes(1, 'big')
                    out_data += ink_fg.to_bytes(1, 'big')
                    oldfg = ink_fg
                
                if full_charset:
                    # swap char indices 0 (square) and 32 (blank space)
                    if char == 0:
                        char = 32
                    elif char == 32:
                        char = 0 
                    if char < 32:
                        out_data += (1).to_bytes(1, 'big')
                else:
                    char += 32
                
                out_data += char.to_bytes(1, 'big')
         
        out_data += end_byte.to_bytes(1, 'big')
        
        # utf-8 is safest encoding to use here, but non-default on Windows
        outfile = open(out_filename, 'wb')
        file_length = 0

        bin_file = open('formats' + os.sep + 'out_cpc_viewer.bin', 'rb') 
        ass_prog = bin_file.read() # reading all lines
        file_length += len(ass_prog)
        bin_file.close

        file_length += len(out_data)

        #actually write file now
        ams=get_AMSDOS_header("joe", "bin", file_length, 0x1000, 0x1000)
        outfile.write(ams)
        outfile.write(bytearray(ass_prog))
        outfile.write(bytearray(out_data))

        outfile.close()

        #call emulator after file has been written
        #os.system("caprice  " + out_filename)
    
        return True
